/*
  L_Standard.inc
  created by Steven L. Reid
  v1.1 31 Aug 97

  This file provides standard definitions used to create realistic
  brick ray traces.

  v1.0 20 Jul 97 File created.
  v1.1 31 Aug 97 Fixed major error in CSD and CSB definition.  Also,
                 added new library include checks!
*/

// POV-Ray Version
#version 3.0

#ifndef (L_Standard)
    #declare L_Standard = true
    // include library!

// Standard Includes (used currently for color defines)
#include "colors.inc"
#include "textures.inc"

// Render in high quality?
#ifndef (l_highq)
    #declare l_highq = true
#end

// Standard dimensions
#declare W   = 0.8                          // Brick width
#declare F   = 0.32                         // Brick flat height
#declare F2  = 2*F                          // Two flats high
#declare H   = 3*F                          // Brick height
#declare E   = F/2                          // Edge thickness
#declare SH  = E                            // Stud height
#declare SD  = W-F                          // Stud diameter
#declare SR  = SD/2                         // Stud radius
#declare CSD = F                            // Cylinder small diameter
#declare CSR = CSD/2                        // Cylinder small radius
#declare CBR = sqrt(F)-SR                   // Cylinder big radius
#declare CBD = 2*CBR                        // Cylinder big diameter
#declare WR  = W/2                          // Width radius (or half width)
#declare C   = 0.025                        // Brick curves radius
#declare G   = 0.001                        // Generic gap
#declare TE  = 0.05                         // Tile edge
#declare CR  = SH//0.09                         // Curve radius (used on head)
#declare ES  = SH/2                         // Edge small
#declare AR  = CSR*sin(radians(45))         // Axle radius
#declare A45 = degrees(atan2((H-SH),W))     // Ramp 45 angle
#declare A25 = degrees(atan2((H-SH),(2*W))) // Ramp 25 angle

/*
 * Standard objects used by most libraries
 */

// Stud
#declare l_stud =
#if (l_highq)
    merge {
        cylinder {-C*z, (SH-C)*z, SR}
        torus {SR-C,C rotate -90*x translate (SH-C)*z}
        cylinder {(SH-C)*z, SH*z, SR-C}
    }
#else
    cylinder { 0, SH*z, SR }
#end

// Stud with hole
#declare l_studh =
#if (l_highq)
    merge {
        difference {
            cylinder { -C*z, (SH-(SR-CSR)/2)*z, SR }
            cylinder { -E*z,            SH*z, CSR }
        }
        torus {
            SR-(SR-CSR)/2,(SR-CSR)/2
            rotate -90*x
            translate (SH-(SR-CSR)/2)*z
        }
    }
#else
    difference {
        cylinder {    0,     SH*z, SR }
        cylinder { -G*z, (SH+G)*z, CSR }
    }
#end

// Stud with hole high
#declare l_studhh =
#if (l_highq)
    merge {
        difference {
            cylinder {    0, (H+SH-(SR-CSR)/2)*z, SR }
            cylinder { -G*z,               2*H*z, CSR }
        }
        torus {
            SR-(SR-CSR)/2, (SR-CSR)/2
            rotate -90*x
            translate (H+SH-(SR-CSR)/2)*z
        }
    }
#else
    difference {
        cylinder {    0, (H+SH)*z, SR }
        cylinder { -G*z,    2*H*z, CSR }
    }
#end

// Cylinder small
#declare l_cs = cylinder { G*z, (H-E)*z, CSR }

// Cylinder small flat
#declare l_csf = cylinder { G*z, (F-E)*z, CSR }

// Cylinder big
#declare l_cb = difference {
    cylinder { G*z, (H-E)*z, CBR}
    cylinder {   0,     H*z, SR}
}

// Cylinder big two high
#declare l_cb2 = difference {
    cylinder { G*z, (2*H-E)*z, CBR}
    cylinder {   0,     2*H*z, SR}
}

// Cylinder big flat
#declare l_cbf = difference {
    cylinder { G*z, (F-E+C)*z, CBR}
    cylinder {   0,     F*z, SR}
}

// Axle Cutout
#declare l_axlecut = difference {
    union {
        box {<-SD,-AR,-H>,<SD,AR,H>}
        box {<-AR,-SD,-H>,<AR,SD,H>}
#if (l_highq)
        box {<-AR-C,-AR-C,-H>,<AR+C,AR+C,H>}
#end
    }
    cylinder {-(H*G)*z, (H+G)*z, SR inverse}
#if (l_highq)
    cylinder {-(H*G)*z, (H+G)*z, C translate <-AR-C,-AR-C,0>}
    cylinder {-(H*G)*z, (H+G)*z, C translate <-AR-C, AR+C,0>}
    cylinder {-(H*G)*z, (H+G)*z, C translate < AR+C,-AR-C,0>}
    cylinder {-(H*G)*z, (H+G)*z, C translate < AR+C, AR+C,0>}
#end
}

/*
 * Standard colors and textures
 */
    // standard LEGO colors
#declare c_white   = color rgb <1.00,1.00,1.00>
#declare c_black   = color rgb <0.10,0.10,0.10>
//#declare c_black   = color rgb <0.15,0.15,0.15> // used with some renders
#declare c_gray    = color rgb <0.75,0.75,0.75>
#declare c_red     = color rgb <1.00,0.00,0.00>
#declare c_green   = color rgb <0.00,0.70,0.00>
#declare c_blue    = color rgb <0.00,0.00,0.85>
#declare c_yellow  = color rgb <1.00,0.95,0.00>
    // transparent LEGO colors
#declare t_clear   = color rgbf  <0.98,0.98,0.98,0.75>
#declare t_cloudy  = color rgbft <0.98,0.98,0.98,0.10,.55>
#declare t_glass   = color rgbft <0.98,0.98,0.98,0.10,.65>
//#declare t_cloudy  = color rgbft <0.85,0.85,0.85,0.10,0.55>
#declare t_red     = color rgbft <0.85,0.00,0.00,0.75,0.10>
#declare t_green   = color rgbft <0.00,0.75,0.00,0.75,0.10>
#declare t_blue    = color rgbft <0.00,0.00,0.85,0.75,0.10>
#declare t_yellow  = color rgbft <0.95,0.95,0.00,0.75,0.10>
#declare t_white   = color rgbt  <1.00,1.00,1.00,1.00>
    // other LEGO colors
#declare c_dkgreen = color rgb <0.14,0.56,0.14>
#declare c_dkgray  = color rgb <0.50,0.50,0.50>
#declare c_brown   = color rgb <0.36,0.25,0.20>
#declare c_ltbrown = color rgb <0.86,0.58,0.44>
#declare c_silver  = color rgb <0.87,0.89,0.96>
#declare c_pink    = color rgb <0.90,0.65,0.65>
#declare c_orange  = color rgb <1.00,0.50,0.00>
    // lego finish
#declare l_finish = finish {
    diffuse 0.5
    ambient 0.3
    phong 0.6
    phong_size 30
//    reflection 0.1
}
#declare l_glass = finish {
    l_finish
    diffuse 0.7
    ambient 0.3
    phong 0.6
    phong_size 30
    reflection 0.1
    refraction 1
    ior 1.5
}
    // bumpy surface for slopes and other bricks
#declare l_bumpy = normal {
    bumps 0.3
    scale 0.03
}
    // textures
#declare l_texture = texture {finish {l_finish}}
#declare l_glasstx = texture {finish {l_glass}}
#declare l_white   = texture {l_texture pigment{c_white}}
#declare l_black   = texture {l_texture pigment{c_black}}
#declare l_gray    = texture {l_texture pigment{c_gray}}
#declare l_red     = texture {l_texture pigment{c_red}}
#declare l_green   = texture {l_texture pigment{c_green}}
#declare l_blue    = texture {l_texture pigment{c_blue}}
#declare l_yellow  = texture {l_texture pigment{c_yellow}}
#declare lt_clear  = texture {l_glasstx pigment{t_clear}}
#declare lt_cloudy = texture {l_glasstx pigment{t_cloudy}}
#declare lt_glass  = texture {l_glasstx pigment{t_glass}}
#declare lt_red    = texture {l_glasstx pigment{t_red}}
#declare lt_green  = texture {l_glasstx pigment{t_green}}
#declare lt_blue   = texture {l_glasstx pigment{t_blue}}
#declare lt_yellow = texture {l_glasstx pigment{t_yellow}}
#declare lt_white = texture {pigment{t_white}}
#declare l_dkgreen = texture {l_texture pigment{c_dkgreen}}
#declare l_dkgray  = texture {l_texture pigment{c_dkgray}}
#declare l_brown   = texture {l_texture pigment{c_brown}}
#declare l_ltbrown = texture {l_texture pigment{c_ltbrown}}
#declare l_silver  = texture {l_texture pigment{c_silver}}
#declare l_pink    = texture {l_texture pigment{c_pink}}
#declare l_orange  = texture {l_texture pigment{c_orange}}

/*
 * Standard cameras and light sources
 */

#end    // end library include
