#!/usr/bin/perl -w

########################################################################
#
# program: blogcal.cgi
# system : Blog Calendar Mod for WebAPP NE (Network Edition)
# author : Copyright (C) 2007 Steven L. Reid
# purpose: Display a calendar of articles (sidebar).
# history:
# 12/02/2006, Steven Reid
#	- script created
# 01/27/2007, Steven Reid
#	- prefixed constants with BC_ to avoid future issues
#
########################################################################

# setup environment
$|++;
#use strict; # turned off due to WebAPP not supporting it

# Read configuration file
BEGIN {
  use CGI::Carp qw(fatalsToBrowser);
  # include initial configuration
  require "$scriptdir/mods/blogcal/data/blogcal-paths.cfg";
  require "$scriptdir/mods/blogcal/data/blogcal.cfg";
  # include language file (if exists)
  if ( -e "$bc_langDir/$userlang") {
    require "$bc_langDir/$userlang";
  } else {
    require "$bc_langDir/$bc_defaultLang";
  }
}

# load libraries
use File::Basename;

# define globals
my @BC_MONTH = split(/\s+/, $bc_msg{'008'});
my @BC_DAY = split(/\s+/, $bc_msg{'009'});
my $BC_URL = "$bc_scriptURL/index.cgi?";

# define variables
my $bc_month; my $bc_day; $bc_year;
my %bc_blog_dates = ();
my $bc_next_year; my $bc_next_month;
my $bc_prev_year; my $bc_prev_month;
my $selfurl = get_self_url();

########################################################################
# Display Blog Calendar
########################################################################
sub display_blog_calendar {
  bc_load_parms();		# load parameters and set variables
  bc_get_blogs();		# get blogs for given month(s)
#  boxtop($bc_title);
  # print the calendar
  bc_print_cal($bc_month, $bc_day, $bc_year, 0);
  # print the dual calendar if requested
  bc_print_cal($bc_prev_month, 0, $bc_prev_year, 1) if $bc_dual_cal;
#  boxbottom();
}
########################################################################

########################################################################
# Load params - get the form variables if need be
########################################################################
sub bc_load_parms {
  # load parameters (set to zero if not passed, or invalid date)
  $bc_month = int(isnum($info{'bc_month'}, 0));
  $bc_month = 0 if $bc_month < 1 || $bc_month > 12;
  $bc_year = int(isnum($info{'bc_year'}, 0));
  $bc_year += 100 if $bc_year < 100;
  $bc_year += 1900 if $bc_year < 1900;
  $bc_day = 0;
  
  # grab today's date
  getTimezones(); # convert to current time
  my ($m, $d, $y) = getNewtime(split(/\s-\s/, $date));
  $y+=100 if $y < 100; $y+=1900;
  
  # fix dates
  if ($bc_month) {
    # set day if month, year are now
    $bc_day = int($d) if $bc_month == int($m) && $bc_year == int($y);
  } else {
    # otherwise, set to today's date
    $bc_month = int($m); $bc_day = int($d); $bc_year = int($y);
  }

  # set prev/next information
  $bc_next_year     = $bc_month == 12 ? $bc_year+1 : $bc_year;
  $bc_prev_year     = $bc_month ==  1 ? $bc_year-1 : $bc_year;
  $bc_next_month    = $bc_month == 12 ?  1 : $bc_month+1;
  $bc_prev_month    = $bc_month ==  1 ? 12 : $bc_month-1;
}
########################################################################

########################################################################
# Get Blogs - gets a list of blogs and sorts them according to date/time
########################################################################
sub bc_get_blogs {
  # walk blog tree, matching any criteria and placing into
  # @blog
  my $m = $bc_month; $m = "0$m" if $m < 10;
  my $y = $bc_year - 1900; $y -= 100 if $y > 100; $y = "0$y" if $y < 10;
  my $pm = $bc_prev_month; $pm = "0$pm" if $pm < 10;
  my $py = $bc_prev_year - 1900;
  $py -= 100 if $py > 100; $py = "0$py" if $py < 10;

  # get list of articles
  opendir (DIR, "$topicsdir/articles");
  my @files = readdir(DIR);
  closedir (DIR);
  @files = grep (/\.txt/, @files);
  
  # walk list of files, looking for matches
  for my $file (@files) {
    open (FILE, "<$topicsdir/articles/$file");
    hold(FILE);
    my $head = <FILE>;
    release(FILE);
    close(FILE);
    my ($adate) = (split ( /\|/, $head))[4];
    # format: 02/25/06 - 22:53:30
    my ($month, $day, $year)
      = getNewtime(split(/\s-\s/, $adate));
    if (($month == $m && $year == $y) ||
        ($month == $pm && $year == $py)) {
      $bc_blog_dates{"$month$day"}->{'day'} =
        "$year$month$day&amp;bc_month=$month&amp;bc_year=$year";
      $bc_blog_dates{"$month$day"}->{'cnt'}++;
    }
  }
}
########################################################################

########################################################################
# Print a nice HTML calendar
########################################################################
sub bc_print_cal {
  # get variables
  my ($month, $day, $year, $dual) = @_;
  my $t = $month; $t = "0$t" if $t < 10;	# used for post test
  $month--;	# make it unix format month (subtract 1)

  # determine leap year
  my $leap = 0;
  $leap = 1 unless $year % 4;
  $leap = 0 unless $year % 100;
  $leap = 1 unless $year % 400;

  # determine length of month
  my $monthlen = 31;
  $monthlen = 28 + $leap if $month == 1;	# February
  $monthlen-- if $month =~ /3|5|8|10/;	# months with 30 days

  # week day of the first of the month using Zeller's Congruence
  my $m = $month + 1; my $d = 1; my $y = $year;
  if ($m < 3) { $m+=12; $y--; }
  my $weekday = ( $d + $m*2 + int(($m+1)*0.6) + 1 + $y +
               int($y/4) - int($y/100) + int($y/400)  ) % 7;

  # initial setup
  $d = undef;		# day of month
  my $c = $weekday-1;	# calendar day of month (used for ends)
  my %WEEKEND = (7=>1,14=>1,21=>1,28=>1,35=>1,42=>1);
  my $next
    = "${selfurl}bc_year=$bc_next_year&amp;bc_month=$bc_next_month";
  my $prev
    = "${selfurl}bc_year=$bc_prev_year&amp;bc_month=$bc_prev_month";

  # display header
  if (!defined &boxtop) { # old themes compatibility
    print qq~<tr><td>~;
  }

  # print top of calendar
  my $title = "$BC_MONTH[$month]  $year";
  $title =~ s/\s/&nbsp;/g;
  print <<END_HTML;
    <div style="$bc_style_blogcal" align="center">
      <p style="$bc_style_title">$title</p>
      <table summary="$title" style="$bc_style_month" align="center">
	<tr>
END_HTML
  for my $day (@BC_DAY) {
    print qq(<th style="$bc_style_week">$day</th>);
  }
  print qq(</tr>\n);

  # print days of month
  print qq(<tr>);
  # print any unused days at start of month
  print qq(<td style="$bc_style_day" bgcolor="$bc_noday_color">&nbsp;</td>) x $weekday;
  # print each day in month
  for $d (1..$monthlen) {
    $c++;
    $zd = ($d < 10 ? "0$d" : $d);
    # is this the end of the week?
    print qq(</tr>\n<tr>) if $WEEKEND{$c};
    # is this a blog, or a regular day?
    if ($bc_blog_dates{"$t$zd"}) {
      my $entry = $bc_blog_dates{"$t$zd"}->{'day'};
      my $cnt = $bc_blog_dates{"$t$zd"}->{'cnt'};
      my $posts = "$cnt "
        . ($cnt == 1 ? $bc_msg{'010'} : $bc_msg{'002'});
      print qq(<td style="$bc_style_daypost" bgcolor="),
        ($d == $day ? "$bc_today_color" :
        $WEEKEND{$c} || $WEEKEND{$c+1} ? "$bc_weekend_color"
	: "$bc_day_color");
      print qq("><a href="${BC_URL}date=$entry"),
        qq( title="$posts">$d</a></td>);
    } else {
      print qq(<td style="$bc_style_day" bgcolor="), ($d == $day ? "$bc_today_color" :
        $WEEKEND{$c} || $WEEKEND{$c+1} ? "$bc_weekend_color"
	: "$bc_day_color");
      print qq(">$d</td>);
    }
  }
  while (not $WEEKEND{++$c}) {
    print qq(<td bgcolor="$bc_noday_color">&nbsp;</td>);
  }
  print qq(</tr><tr>),
    qq(<td bgcolor="$bc_noday_color">&nbsp;</td>) x 7 if $c <= 35;
  print <<END_HTML;
        </tr>
END_HTML
  print <<END_HTML unless $dual;
	<tr>
	  <td colspan="7" style="$bc_style_day">
	    <span style="float: left;"><a href="$prev"
	      style="$bc_style_navigation"
	      title="$bc_msg{'013'}">&lt;&lt;&nbsp;$bc_msg{'011'}</a></span>
	    <span style="float: right;"><a href="$next"
	      style="$bc_style_navigation"
	      title="$bc_msg{'014'}">$bc_msg{'012'}&nbsp;&gt;&gt;</a></span>
	  </td>
	</tr>
END_HTML
  print <<END_HTML;
      </table>
    </div>
END_HTML
  if (!defined &boxbottom) { # old themes compatibility
    print qq~</td></tr>~;
  }
}
########################################################################

########################################################################
sub trim { # trim leading and trailing whitespace
  local $_ = shift || ''; /^\s*(.*?)\s*$/; return $1 }
########################################################################
sub isnum { # check if a number, otherwise load defaults
  my $retval; my $lastval;
  for my $val (@_) {
    $lastval = $val;
    next unless defined $val;
    next unless $val =~ /^\s*(\d+)\s*$/;
    return $1;
  }
  return $lastval;
}
########################################################################
sub get_self_url { # determine self url
  my $uri = $ENV{'REQUEST_URI'} || $BC_URL;
  $uri =~ s/\&*bc_year=\d+//g;
  $uri =~ s/\&*bc_month=\d+//g;
  $uri =~ s/\&/\&amp;/g;
  $uri =~ s/\&amp;amp;/\&amp;/g;
  $uri .= ($uri !~ /\?/ ? '?' : $uri !~ /\?$/ ? '&amp;' : '');
  return $uri;
}
########################################################################

1;
